from PyQt4 import QtCore
from PyQt4 import QtGui
from Definitions import LEDNames
from Action import ActionStart
from Action import Speech
from TextCommunicator import TextCommunicator
from UI.ActionPushButton import ActionPushButton
from UI.SudokuBoard import SudokuBoard
from EmpathyButton import EmpathyButton
from EmpathyComponent import EmpathyComponent
from EmpathySpeech import EmpathySpeech
from EmpathySudoku import SudokuBoards
from MotionList import MotionList
from Text import Text
import Constants
import random

class Empathy(QtGui.QWidget):
    def __init__(self):
        super(Empathy, self).__init__()
        MotionList.initialize()
        random.seed()

        self._actionQueue = None
        self._idleTimer = QtCore.QTimer()
        self._idleTimer.setInterval(10000)
        self._idleTimer.timeout.connect(self.on_idleTimer_timeout)
        self._jitterLevel = 0
        self._nao = None
        self._specials = dict()
        self._texts = dict() # replacing some text in TTS box
        self._textComm = TextCommunicator()
        self._textComm.textAvailable.connect(self.on_textCommunicator_textAvailable)

        self._sudokuCurrSubgrid = None
        self._sudokuLastAnswer = [0, 0, 0] # x y value
        self._sudokuPrevBoard = list()
        for i in range(9):
            self._sudokuPrevBoard.append(list())
            for j in range(9):
                self._sudokuPrevBoard[i].append(0)
                self._sudokuPrevBoard[i][j] = 0
            #END for
        #END for

        self._setupUi()
        self._setupShortcuts()
    #END __init__()

    def __del__(self):
        MotionList.destroy()
    #END __del__()

    def LEDActive(self):
        self._textComm.beginProcess()
        self._nao.LEDrandomEyes(1.0, True)
    #END LEDActive()

    def LEDNormal(self):
        self._textComm.endProcess()
        rgb = 0x00000000
        if self._jitterLevel <= 1:
            rgb = 0x0087ceeb
        elif self._jitterLevel <= 2:
            rgb = 0x0000FF7F
        elif self._jitterLevel <= 3:
            rgb = 0x003CB371
        elif self._jitterLevel <= 4:
            rgb = 0x00228B22
        elif self._jitterLevel <= 5:
            rgb = 0x0000ff00
        else:
            rgb = 0x0087ceeb
        #END if
        self._nao.LEDfadeRGB(LEDNames.Face, rgb, 0.5, True)
        self._nao.LEDfadeRGB(LEDNames.Chest, 0x0000ff00, 0.5, True)
        self._nao.LEDfadeRGB(LEDNames.LeftEar, 0x00ff6100, 0.5, True)
        self._nao.LEDfadeRGB(LEDNames.RightEar, 0x00ff6100, 0.5, True)
    #END LEDNormal()

    def setActionQueue(self, actionQueue):
        if self._actionQueue is not None:
            self._actionQueue.executing.disconnect(self.on_actionQueue_executed)
        #END if
        self._actionQueue = actionQueue
        if self._actionQueue is not None:
            self._actionQueue.executing.connect(self.on_actionQueue_executed)
        #END if
    #END setActionQueue()

    def setNao(self, nao):
        if self._nao is not None:
            self._nao.connected.disconnect(self.on_nao_connected)
            self._nao.disconnected.disconnect(self.on_nao_disconnected)
        #END if
        self._nao = nao
        if self._nao is not None:
            self._nao.connected.connect(self.on_nao_connected)
            self._nao.disconnected.connect(self.on_nao_disconnected)
        #END if
    #END setNao()

    def speech(self, txt, speed, shaping):
        key = txt.lower()
        if self._texts.has_key(key):
            actions = self._texts[key].get(self._jitterLevel)
        else:
            dispTxt = txt
            if self._jitterLevel > 1:
                if random.randint(0, 100) <= 70:
                    txt = txt.replace("want", "wah- wah- want")
                #END if
                if random.randint(0, 100) <= 50:
                    txt = txt.replace("where", "wheh- wheh- where")
                #END if
                if random.randint(0, 100) <= 30:
                    txt = txt.replace("you", Speech.markSpeech(50, 120) + "you- you- you." + Speech.markSpeech(85))
                #END if
            #END if
            actions = self._specials[Constants.BHV_MOVEMENT].get(self._jitterLevel)
            actions.append(Speech(txt, speed, shaping, blocking = False))
            actions.append(Text(dispTxt))
        #END if
        return actions
    #END speech()

    def hideEvent(self, event):
        if self._idleTimer.isActive():
            self._idleTimer.stop()
        #END if
    #END hideEvent()

    def showEvent(self, event):
        # Testing code
        #self._textComm.connectToHost("127.0.0.1")
        #self._textComm.sendName(Constants.ROBOT_NAME)
        pass
    #END showEvent()

    def on_actionbutton_clicked(self):
        if isinstance(self.sender(), ActionPushButton):
            self._actionQueue.addActions(self.sender().getRobotActions())
        elif isinstance(self.sender(), EmpathyButton):
            self._actionQueue.addActions(self.sender().getRobotActions(self._jitterLevel))
        #END if
    #END on_actionbutton_clicked()

    def on_actionQueue_executed(self, action):
        if isinstance(action, Text):
            script = str(action.paramToString())
            if self._jitterLevel <= 0:
                numAdditions = 0
            elif self._jitterLevel <= Constants.MAX_JITTER_LEVEL:
                numAdditions = min(len(script.replace(" ","")) / 2, random.randint(0, self._jitterLevel * 2))
            else:
                numAdditions = 0
            #END if
            for i in range(0, numAdditions):
                rep = ""
                i = 0
                length = random.randint(1, self._jitterLevel)
                while i < length:
                    unit = random.randint(0, len(Constants.TEXT_DISTORTION) - 1)
                    unit = Constants.TEXT_DISTORTION[unit]
                    rep += str(unit)
                    i += 1
                #END while
                loc = random.randint(0, len(script) - 1)
                script = script[:loc] + rep + script[loc:]
            #END for
            self._textComm.send(script)
        #END if
    #END on_actionQueue_executed()

    def on_filler_triggered(self):
        actions = self._specials[Constants.BHV_FILTER].get(self._jitterLevel)
        actions.append(ActionStart())
        self._actionQueue.addActions(actions)
    #END on_filler_triggered()

    def on_idleInterval_valueChanged(self, value):
        self._idleTimer.setInterval(value)
    #END on_idleInterval_valueChanged()

    def on_idleTimer_timeout(self):
        actions = self._specials[Constants.BHV_IDLE_SMALL].get(self._jitterLevel)
        actions.append(ActionStart())
        if not self._actionQueue.isRunning() and self._actionQueue.rowCount(None) <= 0:
            self._actionQueue.addActions(actions)
        #END if
    #END on_idleTimer_timeout()

    def on_idleToggle_changed(self):
        if self._idleTimer.isActive():
            self._idleTimer.stop()
        else:
            interval = self._specials[Constants.CNTL_IDLE_INTERVAL].value()
            self._idleTimer.start(interval)
        #END if
    #END on_idleToggle_changed()

    def on_inputParticipantName_edited(self, value):
        for i in self._specials[Constants.BHV_PARTICIPANT]:
            self._specials[Constants.BHV_PARTICIPANT][i].replace(value)
        #END for
    #END on_inputName_edited()

    def on_inputItemName_edited(self, value):
        for i in self._specials[Constants.BHV_LIKEITEM]:
            self._specials[Constants.BHV_LIKEITEM][i].replace(value)
        #END for
    #END on_inputName_edited()

    def on_jitterLevel_valueChanged(self, value):
        if value <= 0:
            self._jitterLevel = 0
        elif value < Constants.MAX_JITTER_LEVEL:
            self._jitterLevel = value - 1
        else:
            self._jitterLevel = 0
        #END if
    #END on_jitterLevel_valueChanged()

    def on_nao_connected(self):
        self._textComm.connectToHost(self._nao.ipAddress)
        self._textComm.sendName(Constants.ROBOT_NAME)
    #END on_nao_connected()

    def on_nao_disconnected(self):
        self._textComm.disconnectFromHost()
    #END on_nao_disconnected()

    def on_sudokuBoard_deselectSubgrid(self):
        if self._sudokuCurrSubgrid is not None:
            self._wgtSudoku.highlightSubgrid(self._sudokuCurrSubgrid[0], self._sudokuCurrSubgrid[1])
            self._sudokuCurrSubgrid = None
        #END if
    #END on_sudokuBoard_deselectSubgrid()

    def on_sudokuBoard_repeatAnswer(self):
        actions = self._specials[Constants.BHV_SUDOKU_REPEAT_SHORT].get(self._jitterLevel)
        for action in actions:
            if isinstance(action, EmpathySpeech):
                self._sudokuReplaceAnswer(action, self._sudokuLastAnswer[1], self._sudokuLastAnswer[0], self._sudokuLastAnswer[2], False, False)
            #END if
            elif isinstance(action, Text):
                self._sudokuReplaceAnswer(action, self._sudokuLastAnswer[1], self._sudokuLastAnswer[0], self._sudokuLastAnswer[2], True, False)
            #END if
        #END for
        self._actionQueue.addActions(actions)
    #END on_sudokuBoard_repeatAnswer()

    def on_sudokuBoard_repeatAnswerLong(self):
        actions = self._specials[Constants.BHV_SUDOKU_REPEAT_LONG].get(self._jitterLevel)
        for action in actions:
            if isinstance(action, EmpathySpeech):
                self._sudokuReplaceAnswer(action, self._sudokuLastAnswer[1], self._sudokuLastAnswer[0], self._sudokuLastAnswer[2], False, True)
            #END if
            elif isinstance(action, Text):
                self._sudokuReplaceAnswer(action, self._sudokuLastAnswer[1], self._sudokuLastAnswer[0], self._sudokuLastAnswer[2], True, True)
            #END if
        #END for
        self._actionQueue.addActions(actions)
    #END on_sudokuBoard_repeatAnswerLong()

    def on_sudokuBoard_selected(self):
        intValue = int(self.sender().text()) - 1
        if self._sudokuCurrSubgrid is None:
            # selecting subgrid
            x = intValue % 3
            y = 2 - (intValue / 3)
            self._sudokuCurrSubgrid = [y, x]
            self._wgtSudoku.highlightSubgrid(self._sudokuCurrSubgrid[0], self._sudokuCurrSubgrid[1], QtGui.QColor(0, 255, 0))
        else:
            # selecting cell
            x = (self._sudokuCurrSubgrid[1] * 3) + (intValue % 3)
            y = (self._sudokuCurrSubgrid[0] * 3) + (2 - (intValue / 3))
            self._wgtSudoku.highlightSubgrid(self._sudokuCurrSubgrid[0], self._sudokuCurrSubgrid[1])
            self._wgtSudoku.focus(y, x)
            self._sudokuCurrSubgrid = None
        #END if
    #END on_sudokuBoard_selected()

    def on_sudokuBoard_solveOne(self):
        self._wgtSudoku.solveOne()
    #END on_sudokuBoard_solveOne()

    def on_sudokuBoard_valueChanged(self, i, j, value):
        self.on_sudokuBoard_deselectSubgrid()
        self._sudokuLastAnswer = [i, j, value]
        if value != 0:
            actions = self._specials[Constants.BHV_SUDOKU_ANSWER].get(self._jitterLevel)
            for action in actions:
                if isinstance(action, EmpathySpeech):
                    self._sudokuReplaceAnswer(action, j, i, value, False, False)
                #END if
                elif isinstance(action, Text):
                    self._sudokuReplaceAnswer(action, j, i, value, True, False)
                #END if
            #END for
            self._actionQueue.addActions(actions)
        #END if
    #END on_sudokuBoard_valueChanged()

    def on_sudokuBoard_wrongAnswer(self):
        i = random.randint(9, 29)
        j = random.randint(9, 19)
        value = random.randint(10, 30)
        self.on_sudokuBoard_valueChanged(i, j, value)
    #END on_sudokuBoard_wrongAnswer()

    def on_sudokuGamebutton_clicked(self):
        label = str(self.sender().text()[5:])
        if label.isdigit():
            index = int(label)
        else:
            index = 0
        self.on_sudokuBoard_deselectSubgrid()
        for i in range(9):
            for j in range(9):
                self._sudokuPrevBoard[i][j] = self._wgtSudoku.get(i, j)
                self._wgtSudoku.set(i, j, SudokuBoards[index][i][j])
            #END for
        #END for
        self._wgtSudoku.resetLastCoordinate()
    #END on_sudokuGamebutton_clicked()

    def on_sudokuPreviousBoard_clicked(self):
        self.on_sudokuBoard_deselectSubgrid()
        prevBoard = list()
        for i in range(9):
            prevBoard.append(list())
            for j in range(9):
                prevBoard[i].append(0)
                prevBoard[i][j] = self._wgtSudoku.get(i, j)
                self._wgtSudoku.set(i, j, self._sudokuPrevBoard[i][j])
            #END for
        #END for
        self._sudokuPrevBoard = prevBoard
    #END on_sudokuPreviousBoard_clicked()

    def on_textCommunicator_textAvailable(self, text):
        print text
    #END on_textCommunicator_textAvailable()

    def _sudokuReplaceAnswer(self, speech, x, y, answer, display, verbose):
        if display:
            coord = "at " + str(chr(ord('A') + x))
        else:
            coord = "aet, " + Constants.COORDINATE.get(x, str(chr(ord('A') + x)) + ".")
        #END if
        if verbose:
            coord += Constants.COORDINATE_VERBOSE.get(x, "")
        #END if
        coord += str(y + 1)
        speech.replace(coord, str(answer))
    #END _sudokuReplaceAnswer()

    def _setupShortcuts(self):
        for i in range(1, 10):
            action = QtGui.QAction(str(i), self)
            action.setShortcut(QtCore.Qt.Key_0 + i)
            action.triggered.connect(self.on_sudokuBoard_selected)
            self.addAction(action)
        #END for

        action = QtGui.QAction("Toggle_Idle", self)
        action.setShortcut(QtCore.Qt.Key_Minus)
        action.triggered.connect(self.on_idleToggle_changed)
        self.addAction(action)

        action = QtGui.QAction("Execute_Filler", self)
        action.setShortcut(QtCore.Qt.Key_Plus)
        action.triggered.connect(self.on_filler_triggered)
        self.addAction(action)

        action = QtGui.QAction("Sudoku_SolveOne", self)
        action.setShortcut(QtCore.Qt.Key_0)
        action.triggered.connect(self.on_sudokuBoard_solveOne)
        self.addAction(action)

        action = QtGui.QAction("Sudoku_RepeatAnswer", self)
        action.setShortcut(QtCore.Qt.Key_Period)
        action.triggered.connect(self.on_sudokuBoard_repeatAnswer)
        self.addAction(action)

        action = QtGui.QAction("JLv_Increment", self)
        action.setShortcut("Shift+Up")
        action.triggered.connect(lambda: self._specials[Constants.CNTL_JITTERLEVEL].setValue(self._specials[Constants.CNTL_JITTERLEVEL].value() + 1))
        self.addAction(action)

        action = QtGui.QAction("JLv_Decrement", self)
        action.setShortcut("Shift+Down")
        action.triggered.connect(lambda: self._specials[Constants.CNTL_JITTERLEVEL].setValue(self._specials[Constants.CNTL_JITTERLEVEL].value() - 1))
        self.addAction(action)
    #END _setupShortcuts()

    def _setupUi(self):
        splitter = QtGui.QSplitter(self)
        splitter.setOrientation(QtCore.Qt.Horizontal)

        ################################################################################
        # initialize UI components for scenario
        ################################################################################
        components, specials, texts = EmpathyComponent.factoryScenario()
        EmpathyComponent.setupWidget(splitter, components, self.on_actionbutton_clicked)
        self._specials = dict(self._specials.items() + specials.items())
        self._texts = dict(self._texts.items() + texts.items())

        ################################################################################
        # initialize UI components for playing motions
        ################################################################################
        components, specials, texts = EmpathyComponent.factoryMotions()
        EmpathyComponent.setupWidget(splitter, components, self.on_actionbutton_clicked)
        self._specials = dict(self._specials.items() + specials.items())
        self._texts = dict(self._texts.items() + texts.items())

        ################################################################################
        # initialize UI components for interaction behaviors
        ################################################################################
        components, specials, texts = EmpathyComponent.factoryInteractions()
        EmpathyComponent.setupWidget(splitter, components, self.on_actionbutton_clicked)
        self._specials = dict(self._specials.items() + specials.items())
        self._texts = dict(self._texts.items() + texts.items())

        ################################################################################
        # initialize UI components for sudoku play
        ################################################################################
        self._wgtSudoku = SudokuBoard()
        self._wgtSudoku.valueChanged.connect(self.on_sudokuBoard_valueChanged)

        splitterSudoku = QtGui.QSplitter()
        splitterSudoku.setOrientation(QtCore.Qt.Vertical)
        splitterSudoku.addWidget(self._wgtSudoku)

        widgetControls = QtGui.QWidget(splitterSudoku)
        layoutControls = QtGui.QHBoxLayout(widgetControls)
        layoutControls.setMargin(0)

        widgetGames = QtGui.QWidget(widgetControls)
        layoutGames = QtGui.QVBoxLayout(widgetGames)
        layoutGames.setMargin(0)
        button = QtGui.QPushButton("Prev. Board", widgetGames)
        button.clicked.connect(self.on_sudokuPreviousBoard_clicked)
        layoutGames.addWidget(button)
        button = QtGui.QPushButton("Training", widgetGames)
        button.clicked.connect(self.on_sudokuGamebutton_clicked)
        layoutGames.addWidget(button)
        for i in range(1, len(SudokuBoards)):
            button = QtGui.QPushButton("Game " + str(i), widgetGames)
            button.clicked.connect(self.on_sudokuGamebutton_clicked)
            layoutGames.addWidget(button)
        #END for
        scrollGames = QtGui.QScrollArea()
        scrollGames.setAlignment(QtCore.Qt.AlignCenter)
        scrollGames.setWidget(widgetGames)
        layoutScrollGames = QtGui.QHBoxLayout()
        layoutScrollGames.setMargin(0)
        layoutScrollGames.addWidget(scrollGames)

        components, specials, texts = EmpathyComponent.factorySudokuUi()
        widget = EmpathyComponent.setupWidget(widgetControls, components, self.on_actionbutton_clicked)
        self._specials = dict(self._specials.items() + specials.items())
        self._texts = dict(self._texts.items() + texts.items())

        layoutControls.addLayout(layoutScrollGames)
        layoutControls.addWidget(widget)
        layoutSudoku = QtGui.QHBoxLayout(splitter)
        layoutSudoku.setMargin(0)
        layoutSudoku.addWidget(splitterSudoku)

        ################################################################################
        # initialize main UI layout
        ################################################################################
        layout = QtGui.QHBoxLayout(self)
        layout.setMargin(0)
        layout.addWidget(splitter)

        self._specials[Constants.CNTL_PARTICIPANT_NAME].textEdited.connect(self.on_inputParticipantName_edited)
        self._specials[Constants.CNTL_JITTERLEVEL].valueChanged.connect(self.on_jitterLevel_valueChanged)
        self._specials[Constants.CNTL_LIKEITEM_NAME].textEdited.connect(self.on_inputItemName_edited)
        self._specials[Constants.CNTL_IDLE_INTERVAL].valueChanged.connect(self.on_idleInterval_valueChanged)
        self._specials[Constants.CNTL_SUDOKU_SOLVE].clicked.connect(self.on_sudokuBoard_solveOne)
        self._specials[Constants.CNTL_SUDOKU_REPEAT_SHORT].clicked.connect(self.on_sudokuBoard_repeatAnswer)
        self._specials[Constants.CNTL_SUDOKU_REPEAT_LONG].clicked.connect(self.on_sudokuBoard_repeatAnswerLong)
        self._specials[Constants.CNTL_SUDOKU_WRONG_ANSWER].clicked.connect(self.on_sudokuBoard_wrongAnswer)
    #END _setupUi()
#END Empathy
