from PyQt4 import QtCore
from PyQt4 import QtNetwork
import time
import struct

class TextCommunicator(QtCore.QObject):
    _HEADER = b'TEXT'
    _CHAT_NAME = "\\NAME\\"
    _PROCESS_BEGIN = "\\PROCESS_BEGIN\\"
    _PROCESS_END = "\\PROCESS_END\\"

    def __init__(self):
        super(TextCommunicator, self).__init__()
        self._lastText = ""
        self._socket = None
        self._recved = QtCore.QByteArray()
    #END __init__()

    def __del__(self):
        self.disconnectFromHost()
    #END __del__()

    textAvailable = QtCore.pyqtSignal(str)

    def connectToHost(self, ipAddress, port = 9558):
        self._socket = QtNetwork.QTcpSocket(self)
        self._socket.readyRead.connect(self.on__socket_readyRead)
        self._socket.connectToHost(ipAddress, port)
        if not self._socket.waitForConnected(5000):
            self._socket.disconnectFromHost()
            self._socket = None
        #END if
    #END connectToHost()

    def disconnectFromHost(self):
        if self._socket is not None:
            self._socket.disconnectFromHost()
            self._socket = None
        #END if
        self._recved.clear()
    #END disconnectFromHost()

    def beginProcess(self):
        self.send(TextCommunicator._PROCESS_BEGIN)
    #END beginProcess()

    def endProcess(self):
        self.send(TextCommunicator._PROCESS_END)
    #END endProcess()

    def send(self, text):
        if self._socket is not None and text != "":
            packet = QtCore.QByteArray()
            packet.append(TextCommunicator._HEADER)
            packet.append(struct.pack("I", len(text) + 8))
            packet.append(text.encode('ascii'))
            self._socket.write(packet)
            if self._lastText != text:
                self._lastText = text
                self.textAvailable.emit("< " + TextCommunicator.timeMarker() + " " + text)
            #END if
        #END if
    #END send

    def sendName(self, name):
        self.send(TextCommunicator._CHAT_NAME + name + " ")
    #END sendName()

    def on__socket_readyRead(self):
        packet = self._socket.readAll()
        self._recved.append(packet)
        processing = True
        while processing and self._recved.length() > len(TextCommunicator._HEADER):
            if self._recved.startsWith(TextCommunicator._HEADER):
                # header is found
                if self._recved.length() < 8:
                    # wait for more bytes to come
                    processing = False
                else:
                    length = self._recved.mid(4, 4)
                    length = struct.unpack("I", length.data())[0]
                    if self._recved.length() < length:
                        # wait for more bytes to come
                        processing = False
                    else:
                        text = self._recved.mid(8, length - 8)
                        text = text.data()
                        self._recved.remove(0, length)
                        self.textAvailable.emit("> " + TextCommunicator.timeMarker() + " " + text)
                    #END if
                #END if
            else:
                # move to text byte to find header
                self._recved.remove(0, 1)
            #END if
        #END while
    #END on__socket_readyRead()

    @staticmethod
    def timeMarker():
        return "[" + time.strftime("%H:%M:%S", time.localtime()) + "]"
    #END timeMarker()
#END class
